﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI.WebControls;

using Telerik.Web.UI;

using VeteransAffairs.Registries.BusinessManager;

public partial class PatientImageDetails : ImagingBasePage
{
    private readonly UserAccountManager _currentUser;
    private readonly EFRPatientManager _patientManager;
    private readonly PatientImageDetailManager _patientImageDetailManager;

    private static readonly IEnumerable<string> _referenceImageBodyParts;
    private static readonly IEnumerable<string> _referenceImageFragmentCounts;
    private static readonly IEnumerable<string> _referenceImageReasons;
    private static readonly IEnumerable<string> _referenceImageTypes;

    static PatientImageDetails()
    {
        _referenceImageBodyParts = PatientImageDetailManager.LoadReferenceImageBodyParts();
        _referenceImageFragmentCounts = PatientImageDetailManager.LoadReferenceImageFragmentCounts();
        _referenceImageReasons = PatientImageDetailManager.LoadReferenceImageReasons();
        _referenceImageTypes = PatientImageDetailManager.LoadReferenceImageTypes();
    }

    public PatientImageDetails()
    {
        _currentUser = (UserAccountManager)HttpContext.Current.User;
        _patientManager = new EFRPatientManager();
        _patientImageDetailManager = new PatientImageDetailManager();

        _patientImageDetailManager.BOSaveSuccess += new BaseBO.BOSaveSuccessEvent(manager_BOSaveSuccess);
    }

    protected void Page_Load(object sender, EventArgs e)
    {
        if (ViewState[OriginallyNavigatedFrom] == null)
            ViewState[OriginallyNavigatedFrom] = Request.UrlReferrer;

        if (!IsPostBack)
        {
            int patientId;
            var patientIdPresent = int.TryParse(Request.QueryString["patientId"], out patientId);

            int patientImageId;
            var patientImageIdPresent = int.TryParse(Request.QueryString["patientImageId"], out patientImageId);

            if (patientIdPresent && patientImageIdPresent)
            {
                PopulatControls();

                LoadData(patientImageId);
            }
        }

        RadComboBoxImageType.Focus();
    }

    protected void SaveButton_Click(object sender, EventArgs e)
    {
        if (Page.IsValid)
        {
            int patientId;
            var patientIdPresent = int.TryParse(Request.QueryString["patientId"], out patientId);

            int patientImageId;
            var patientImageIdPresent = int.TryParse(Request.QueryString["patientImageId"], out patientImageId);

            if (patientIdPresent && patientImageIdPresent)
            {
                var detail = _patientImageDetailManager.GetPatientImageDetails(patientImageId);
                if (detail != null)
                {
                    _patientImageDetailManager.UpdatePatientImage(
                    GetUpdateParameters(detail.StudyId),
                    GetCheckedItemsIndexList(RadComboBoxBodyPartImaged),
                    GetCheckedItemsIndexList(RadComboBoxImagingReason));

                    LoadData(patientImageId);
                }
            }
        }
    }

    private object[] GetUpdateParameters(string study)
    {
        return new object[]
        {
            study,
            GetRadComboBoxSelectedIndex(RadComboBoxNumberOfFragments),
            GetRadComboBoxSelectedIndex(RadComboBoxImageType),
            GetRadComboBoxBooleanValue(RadComboBoxFragmentVerified),
            GetRadComboBoxBooleanValue(RadComboBoxReviewed),
            GetRadComboBoxBooleanValue(RadComboBoxIncludeInReport),
            GetValueFromRadTextBox(RadTextBoxSmallestFragmentSize),
            GetValueFromRadTextBox(RadTextBoxLargestFragmentSize),
            GetValueFromRadTextBox(RadTextBoxImageComments),
            GetValueFromRadTextBox(RadTextBoxOtherImageReason),
            GetValueFromRadTextBox(RadTextBoxOtherImageType),
            _currentUser.UserName
        };
    }

    private IReadOnlyList<int> GetCheckedItemsIndexList(RadComboBox combo)
    {
        List<int> items = null;
        var checkedItems = combo.CheckedItems;
        if (checkedItems != null)
            items = checkedItems.Select(i => i.Index).ToList();

        return items;
    }

    private void LoadData(int patientImageId)
    {
        var patientSummary = GetOrStorePatientSummary();

        SetBreadCrumbLabel(patientSummary);

        var detail = _patientImageDetailManager.GetPatientImageDetails(patientImageId);
        if (detail != null)
        {
            GetReadOnlyValues(detail);

            GetSelectedImageType(detail);
            GetSelectedFragmentCount(detail);

            GetCheckedImageBodyParts(patientImageId);
            GetCheckedImageReasons(patientImageId);

            LoadOtherControls(detail);
        }

        // we do this so that saved changes are visible when we go "back"
        BackButton.PostBackUrl = $"PatientImaging.aspx?patientId={patientSummary.Id}";
    }

    private void GetReadOnlyValues(PatientImageDetail detail)
    {
        LabelStudyDescription.Text = detail.StudyDescription;

        if (detail.SystemIdentified.HasValue)
            LabelSystemIDFragment.Text = detail.SystemIdentified.Value ? "Yes" : "No";
    }

    private void SetBreadCrumbLabel(PatientSummary patientSummary)
    {
        BreadCrumbLabel.Text = $"{BreadCrumbLabel.Text}{patientSummary.FullName}";
        LabelPatientName.Text = patientSummary.FullName;
        LabelPatientSnum.Text = patientSummary.Snum;
    }

    private void GetCheckedImageBodyParts(int patientImageId)
    {
        RadComboBoxBodyPartImaged.ClearCheckedItems();

        var bodyParts = _patientImageDetailManager.GetImageBodyParts(patientImageId);
        if (bodyParts != null && bodyParts.Any())
        {
            foreach (var bp in bodyParts)
                RadComboBoxBodyPartImaged.FindItemByText(bp.Description).Checked = true;
        }
    }

    private void GetCheckedImageReasons(int patientImageId)
    {
        RadComboBoxImagingReason.ClearCheckedItems();

        var imageReasons = _patientImageDetailManager.GetImageReasons(patientImageId);
        if (imageReasons != null && imageReasons.Any())
        {
            foreach (var reason in imageReasons)
                RadComboBoxImagingReason.FindItemByText(reason.Description).Checked = true;
        }
    }

    private void GetSelectedImageType(PatientImageDetail detail)
    {
        RadComboBoxImageType.ClearSelection();

        if (detail.ImageTypeId.HasValue)
            RadComboBoxImageType.FindItem(i => i.Index == detail.ImageTypeId.Value).Selected = true;
    }

    private void GetSelectedFragmentCount(PatientImageDetail detail)
    {
        RadComboBoxNumberOfFragments.ClearSelection();

        if (detail.ImageFragmentCountId.HasValue)
            RadComboBoxNumberOfFragments.FindItem(i => i.Index == detail.ImageFragmentCountId.Value).Selected = true;
    }

    private void LoadOtherControls(PatientImageDetail detail)
    {
        if (detail.FragmentVerfied.HasValue)
            RadComboBoxFragmentVerified.SelectedIndex = detail.FragmentVerfied.Value ? 1 : 2;
        else
            RadComboBoxFragmentVerified.SelectedIndex = 0;

        if (detail.IncludeInReport.HasValue)
            RadComboBoxIncludeInReport.SelectedIndex = detail.IncludeInReport.Value ? 1 : 2;
        else
            RadComboBoxIncludeInReport.SelectedIndex = 0;

        if (!string.IsNullOrEmpty(detail.LargestFragmentSize))
            RadTextBoxLargestFragmentSize.Text = detail.LargestFragmentSize.ToString();
        else
            RadTextBoxLargestFragmentSize.Text = string.Empty;

        if (!string.IsNullOrEmpty(detail.SmallestFragmentSize))
            RadTextBoxSmallestFragmentSize.Text = detail.SmallestFragmentSize.ToString();
        else
            RadTextBoxSmallestFragmentSize.Text = string.Empty;

        if (detail.Reviewed.HasValue)
            RadComboBoxReviewed.SelectedIndex = detail.Reviewed.Value ? 1 : 2;
        else
            RadComboBoxReviewed.SelectedIndex = 0;

        if (detail.Comments != null)
            RadTextBoxImageComments.Text = detail.Comments;
        else
            RadTextBoxImageComments.Text = string.Empty;

        if (detail.ImageReasonOtherText != null)
            RadTextBoxOtherImageReason.Text = detail.ImageReasonOtherText;
        else
            RadTextBoxOtherImageReason.Text = string.Empty;

        if (detail.ImageTypeOtherText != null)
            RadTextBoxOtherImageType.Text = detail.ImageTypeOtherText;
        else
            RadTextBoxOtherImageType.Text = string.Empty;
    }

    private void PopulatControls()
    {
        PopulateBooleanRadComboBoxes();
        PopulateSpecializedRadComboBoxes();
    }

    private void PopulateBooleanRadComboBoxes()
    {
        foreach (var combo in GetBooleanRadComboBoxes())
        {
            combo.Items.Add(new RadComboBoxItem("(Select a Value)"));
            combo.Items.Add(new RadComboBoxItem("Yes"));
            combo.Items.Add(new RadComboBoxItem("No"));
        }
    }

    private void PopulateSpecializedRadComboBoxes()
    {
        RadComboBoxImageType.Items.Add(new RadComboBoxItem("(Select a Value)"));
        foreach (var item in _referenceImageTypes)
            RadComboBoxImageType.Items.Add(new RadComboBoxItem(item));

        foreach (var item in _referenceImageReasons)
            RadComboBoxImagingReason.Items.Add(new RadComboBoxItem(item));

        foreach (var item in _referenceImageBodyParts)
            RadComboBoxBodyPartImaged.Items.Add(new RadComboBoxItem(item));

        RadComboBoxNumberOfFragments.Items.Add(new RadComboBoxItem("(Select a Value)"));
        foreach (var item in _referenceImageFragmentCounts)
            RadComboBoxNumberOfFragments.Items.Add(new RadComboBoxItem(item));
    }

    private IEnumerable<RadComboBox> GetBooleanRadComboBoxes()
    {
        return new List<RadComboBox>
        {
            RadComboBoxFragmentVerified,
            RadComboBoxReviewed,
            RadComboBoxIncludeInReport
        };
    }

    private bool? GetRadComboBoxBooleanValue(RadComboBox combo)
    {
        return combo.SelectedIndex > 0 ? combo.SelectedIndex == 1 : (bool?)null;
    }

    private int? GetRadComboBoxSelectedIndex(RadComboBox combo)
    {
        int? result = null;

        if (combo.SelectedIndex > 0)
            result = combo.SelectedIndex;

        return result;
    }

    private string GetValueFromRadTextBox(RadTextBox textBox)
    {
        return string.IsNullOrEmpty(textBox.Text) ? null : textBox.Text;
    }

}